/** @file   player.h
 * @brief   Declaration of Player - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_PLAYER_H
#define H_WWW_PLAYER_H

// #include <allegro.h>
#include "gameobject.h"
#include "weapon.h"
#include <fstream>


namespace WeWantWar {

// Just let the compiler know about these classes
class BonusObject;
class PlayerController;
class LightBeam;

/** @class  Player
 * @brief   This class represents the human player.
 * @author  Tomi Lamminsaari
 *
 * The GameObject that is controlled by human is an instance of this class.
 *
 * The controllers are handled inside this class. In future the handling of
 * keyboard/mouse/joystick controls will be moved to PlayerController-class.
 */
class Player : public GameObject
{
public:
  ///
  /// Static members and methods
  /// ==========================

  /** Forward movement vector. */
  static eng2d::Vec2D forwardVec;
  /** Backward movement vector. */
  static eng2d::Vec2D backwardVec;
  /** Strafe left movement vector. */
  static eng2d::Vec2D leftVec;
  /** Strage right movement vector. */
  static eng2d::Vec2D rightVec;
  
  /** Forward movement vector. */
  static eng2d::Vec2D forwardSpeed;
  /** Strafe movement vector. */
  static eng2d::Vec2D strafeSpeed;
  /** Backward movement vector. */
  static eng2d::Vec2D backwardSpeed;
  
  
  /** The delay it takes to change weapon */
  static const int WEAPON_CHANGE_DELAY;
  
  /** Index of the reloading delay timer */
  static const int RELOAD_COUNTER_INDEX = 0;
  /** Index of the invulnerability timer */
  static const int INVULNERABILITY_COUNTER_INDEX = 1;
  /** Index of the NO_ENTER counter. This is the delay when we cannot
   * re-enter the vehicle.
   */
  static const int NOENTER_COUNTER_INDEX = 2;

  /** Index of the Rifle's pipe controlpoint */
  static const int RIFLE_CTRLPOINT = 0;
  /** Index of the Uzi 1 pipe's controlpoint. */
  static const int UZI_1_CTRLPOINT = 1;
  /** Index of the Uzi 2 pipe's controlpoint */
  static const int UZI_2_CTRLPOINT = 2;
  /** Index of the controlpoint that tells where the used ammunition
   * jump out of the weapon.
   */
  static const int AMMOS_OUT_CTRLPOINT = 3;
  
  /** Index of the controlpoint for uzi 1 rejecting ammos. */
  static const int UZIAMMOS_OUT_CTRLPOINT1 = 4;
  /** Index of the controlpoint for uzi 2 rejecting ammos. */
  static const int UZIAMMOS_OUT_CTRLPOINT2 = 5;
  
  /** Index of the controlpoint for Rocket Launcher's nose */
  static const int ROCKETLAUNCHER_NOSE_CTRLPOINT = 6;
  /** Index of the controlpoint for the rocketlauncher's launching flame */
  static const int ROCKETLAUNCHER_SHOOTFLAME_CTRLPOINT = 7;
  
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Contructor.
   */
  Player();
  
  /** Destructor.
   */
  virtual ~Player();
  


  ///
  /// Methods
  /// =======


  /** Updates the player's data such as position, some counter, etc.
   */
  virtual void update();
  
  /** Draws the player.
   * @param     pQueue            Pointer to redraw queue
   */
  virtual void redraw( RedrawQueue* pQueue );
  
  /** Checks if given bullet causes us any damage.
   * @param     pB                Pointer to bullet that hit us.
   * @return    'true' if it really did hit us. 'false' if it
   *            went through.
   */
  virtual bool hitByBullet( Bullet* pB );
  
  /** Causes damage if invulnerability is off.
   * @param     points            Number of damagepoints
   * @return    <code>true</code> if we died
   */
  virtual bool causeDamage( int points );
  
  /** Causes damage if invulnerability is off.
   * @param     b                 Pointer to bullet that hit us.
   * @return    <code>true</code> if we died
   */
  virtual bool causeDamage( Bullet* b );
  
  /** Kills the player.
   */
  virtual void kill();
  
  /** Makes sound.
   * @param     id                ID of the sound we should make.
   */
  virtual void makeSound( GameObject::SoundID id ) const;
  
  /** Provides a bonusitem to player.
   * @param     b                 Pointer to the BonusObject.
   * @return    'true' if we picked that bonusitem up. 'false' if we didn't
   *            (for instance our ammocount is full)
   */
  virtual bool pickupBonus( BonusObject* b );
  
  /** Picks up the weapon bonus item. This is called by the 'pickupBonus()'-
   * method.
   * @param     w                 ID of the weapon. See Weapon-class.
   * @return    'true' if we picked it up.
   */
  virtual bool pickWeapon(int w);
  
  /** Writes the player's data to given output stream
   * @param     fout              The output stream
   */
  virtual void savePlayerData( std::ofstream& fout ) const;
  
  /** Reads the player's data from given input stream.
   * @param     fin               The inputstream.
   */
  virtual void loadPlayerData( std::ifstream& fin );

  /** Wakes up the dead player
   */
  virtual void resurrect();

  /** Sets the vehicle object.
   * @param     pObj              Pointer to GameObject or null-pointer to
   *                              exit current vehicle.
   */
  virtual void setVehicle( GameObject* pObj );


  ///
  /// Getter-methods
  /// ==============
  
  /** Returns the type of this object.
   * @return    ObjectID::TYPE_PLAYER
   */
  virtual ObjectID::Type objectType() const;
  
  /** Returns the ID of the weapon the we currently use.
   * @return    ID of the weapon. See Weapon-class for the possible weapon
   *            IDs.
   */
  virtual int getWeapon() const;
  
  /** Returns the ammocount of current weapon.
   * @return    The ammocount
   */
  virtual int getAmmunition() const;
  
  /** Returns the number of grenades we have.
   * @return    Number of grenades.
   */
  virtual int getGrenadeCount() const;
  
  /** Tells if we're reloading our weapon.
   * @return    'true' if we're reloading.
   */
  virtual bool reloading() const;
  
  /** Returns the vehicle-object
   * @return    Pointer to object that is our vehicle.
   */
  virtual GameObject* getVehicle() const;
  
  /** Tells if player is in sniper mode.
   * @return  <code>true</code> if we're in sniper mode.
   */
  bool sniperMode() const;
  
  /** Returns the amount of shield the player has.
   * @param   aShield           Current amount of bodyarmor goes here.
   * @param   aMaxShield        Maximum amount of bodyarmor goes here.
   */
  void getShieldValues( float& aShield, float& aMaxShield ) const;
  
  /** Returns the Flashlight lightbeam
   * @return  Pointer to flashlight's lightbeam.
   */
  LightBeam* FlashLightBeam() const;

protected:

  ///
  /// Protected interface
  /// ===================
  
  /** Requests the mainprogram to show weaponguide
   */
  void requestWeaponguide( int wid );
  
  /** Moves the player when using CONTROLMODE1. See Settings - class.
   */
  void handleControls();
  
  /** Moves the player when using CONTROLMODE2. See Settings - class.
   */
  void handleControls2();
  
  /** Handles the player controlling when we are in sniper mode. In sniper
   * mode the player cannot move but only aim freely.
   * @param   aController         Pointer to PlayerController we're using.
   */
  void handleSniperModeControls( PlayerController* aController );
  
  /** Handles the shooting. Checks the ammocount and spawns right type of
   * bullets.
   */
  void handleShooting();
  
  /** Shoots with rifle. Called by 'handleShooting()'-method.
   */
  void shootRiffle();
  
  /** Shoots with shotgun. Called by 'handleShooting()'-method.
   */
  void shootShotgun();
  
  /** Hits with crowbar. Called by 'handleShooting()'-method.
   */
  void hitHammer();
  
  /** Shoots with flamethrower. Called by 'handleShooting()'-method.
   */
  void shootFlamethrower();
  
  /** Shoots with minigun. Called by 'handleShooting()'-method.
   */
  void shootMinigun();
  
  /** Shoots with uzi. Called by 'handleShooting()'-method.
   */
  void shootUzi();
  
  /** Shoots with sniper rifle.
   */
  void shootSniperRifle();
  
  /** Shoots with rocket launcher.
   */
  void shootRocketLauncher();
  
  /** Throws grenade.
   */
  void throwGrenade();
  
  /** Picks up the correct animation when using CONTROLMODE1.
   */
  void manageAnimations();
  
  /** Changes the weapon but only if we have that weapon.
   * @param     w                 ID of the weapon. See Weapon - class.
   */
  void changeWeapon(int w);
  
  /** Moves player up (CONTROLMODE2)
   */
  void moveUp();
  
  /** Moves player up right (CONTROLMODE2)
   */
  void moveUpRight();
  
  /** Moves player right (CONTROLMODE2)
   */
  void moveRight();
  
  /** Moves player down right (CONTROLMODE2)
   */
  void moveDownRight();
  
  /** Moves player down (CONTROLMODE2)
   */
  void moveDown();
  
  /** Moves player down left (CONTROLMODE2)
   */
  void moveDownLeft();
  
  /** Moves player left (CONTROLMODE2)
   */
  void moveLeft();
  
  /** Moves player up left (CONTROLMODE2)
   */
  void moveUpLeft();
  

  /** Moves the player and sets correct animation (CONTROLMODE2)
   * @param     rM                Movement vector.
   * @param     animID            ID of the animation. See GameAnims - class.
   */
  void moveAndSetAnim(const eng2d::Vec2D& rM, int animID );


  /** Checks, if we're touching a door. If that's the case, we open the door.
   */
  void checkDoors();

  /** When weaponchange keys are pressed, this method converts the controller
   * status values to Weapon::WeaponID values.
   * @param   aKeyValue           Number of the weaponchange key that was
   *                              pressed.
   * @return  WeaponID that matches with weapon change key.
   */
  Weapon::WeaponID getSelectedWeaponId( int aKeyValue ) const;

  /** Selects the next available weapon.
   */
  void selectNextWeapon();
  
  /** Selects the previous available weapon.
   */
  void selectPreviousWeapon();
  
  
  ///
  /// Members
  /// =======
  
  /** The current weapon number. The possible weapon numbers can be found
   * from Consts-class. FOr example, Consts::W_SHOTGUN
   */
  Weapon      m_weapon;
  
  /** Number of grenades the player has */
  int         m_grenades;
  
  /** A shield that protects the player. */
  float       m_shield;
  
  /** This vector contains the bullet count for each weapon. It can be indexed
   * with the weapon constants found in Consts-class ( e.g. Consts::W_RIFLE ).
   * If this vector contains value -1 for some weapon, it means that player
   * does not have that weapon yet.
   */
  std::vector<int>  m_weaponList;
  
  /** When shooting with 2 uzis, the bullets are shot alternaly from
   * both guns. This member tells from wich gun the next bullet should be
   * released.
   */
  int        m_activeUzi;
   
  /** Pointer to GameObject that is our vehicle.
   */
  GameObject* m_pVehicle;
  
  /** Sniper-mode flag */
  bool m_sniperMode;
  
  /** Pointer to flash lgith light beam. */
  LightBeam* iFlashLight;
  
private:
  Player(const Player& rO);
  Player& operator = (const Player& rO);
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: player.h,v $
 * Revision 1.4  2006/06/17 21:45:36  lamminsa
 * Player has flashlight for night levels.
 *
 * Revision 1.3  2006/04/12 21:11:47  lamminsa
 * getShieldValue() method added.
 *
 * Revision 1.2  2006/04/03 20:55:41  lamminsa
 * m_shield member added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.2  2006-01-10 22:59:15+02  lamminsa
 * selectNextWeapon() and selectPreviousWeapon() methods added.
 *
 * Revision 1.1  2005-11-13 14:35:39+02  lamminsa
 * RocketLauncher weapon added.
 *
 */
 
